<?php

namespace Drupal\views\Plugin\views\pager;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\PluginBase;

/**
 * @defgroup views_pager_plugins Views pager plugins
 * @{
 * Plugins to handle paging in views.
 *
 * Pager plugins take care of everything regarding pagers, including figuring
 * out the total number of items to render, setting up the query for paging,
 * and setting up the pager.
 *
 * Pager plugins extend \Drupal\views\Plugin\views\pager\PagerPluginBase. They
 * must be annotated with \Drupal\views\Annotation\ViewsPager annotation,
 * and they must be in namespace directory Plugin\views\pager.
 *
 * @ingroup views_plugins
 * @see plugin_api
 */

/**
 * Base class for views pager plugins.
 */
abstract class PagerPluginBase extends PluginBase {

  public $current_page = NULL;

  public $total_items = 0;

  /**
   * {@inheritdoc}
   */
  protected $usesOptions = TRUE;

  /**
   * Get how many items per page this pager will display.
   *
   * All but the leanest pagers should probably return a value here, so
   * most pagers will not need to override this method.
   */
  public function getItemsPerPage() {
    return isset($this->options['items_per_page']) ? $this->options['items_per_page'] : 0;
  }

  /**
   * Set how many items per page this pager will display.
   *
   * This is mostly used for things that will override the value.
   */
  public function setItemsPerPage($items) {
    $this->options['items_per_page'] = $items;
  }

  /**
   * Get the page offset, or how many items to skip.
   *
   * Even pagers that don't actually page can skip items at the beginning,
   * so few pagers will need to override this method.
   */
  public function getOffset() {
    return isset($this->options['offset']) ? $this->options['offset'] : 0;
  }

  /**
   * Set the page offset, or how many items to skip.
   */
  public function setOffset($offset) {
    $this->options['offset'] = $offset;
  }

  /**
   * Get the current page.
   *
   * If NULL, we do not know what the current page is.
   */
  public function getCurrentPage() {
    return $this->current_page;
  }

  /**
   * Set the current page.
   *
   * @param $number
   *   If provided, the page number will be set to this. If NOT provided,
   *   the page number will be set from the global page array.
   */
  public function setCurrentPage($number = NULL) {
    if (!is_numeric($number) || $number < 0) {
      $number = 0;
    }
    $this->current_page = $number;
  }

  /**
   * Get the total number of items.
   *
   * If NULL, we do not yet know what the total number of items are.
   */
  public function getTotalItems() {
    return $this->total_items;
  }

  /**
   * Get the pager id, if it exists
   */
  public function getPagerId() {
    return isset($this->options['id']) ? $this->options['id'] : 0;
  }

  /**
   * Provide the default form form for validating options
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) { }

  /**
   * Provide the default form form for submitting options
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) { }

  /**
   * Return a string to display as the clickable title for the
   * pager plugin.
   */
  public function summaryTitle() {
    return $this->t('Unknown');
  }

  /**
   * Determine if this pager actually uses a pager.
   *
   * Only a couple of very specific pagers will set this to false.
   */
  public function usePager() {
    return TRUE;
  }

  /**
   * Determine if a pager needs a count query.
   *
   * If a pager needs a count query, a simple query
   */
  public function useCountQuery() {
    return TRUE;
  }

  /**
   * Execute the count query, which will be done just prior to the query
   * itself being executed.
   */
  public function executeCountQuery(&$count_query) {
    $this->total_items = $count_query->execute()->fetchField();
    if (!empty($this->options['offset'])) {
      $this->total_items -= $this->options['offset'];
    }

    return $this->total_items;
  }

  /**
   * If there are pagers that need global values set, this method can
   * be used to set them. It will be called after the query is run.
   */
  public function updatePageInfo() {

  }

  /**
   * Modify the query for paging
   *
   * This is called during the build phase and can directly modify the query.
   */
  public function query() { }

  /**
   * Perform any needed actions just prior to the query executing.
   */
  public function preExecute(&$query) { }

  /**
   * Perform any needed actions just after the query executing.
   */
  public function postExecute(&$result) { }

  /**
   * Perform any needed actions just before rendering.
   */
  public function preRender(&$result) { }

  /**
   * Return the renderable array of the pager.
   *
   * Called during the view render process.
   *
   * @param $input
   *   Any extra GET parameters that should be retained, such as exposed
   *   input.
   */
  public function render($input) { }

  /**
   * Determine if there are more records available.
   *
   * This is primarily used to control the display of a more link.
   */
  public function hasMoreRecords() {
    return $this->getItemsPerPage()
      && $this->total_items > (intval($this->current_page) + 1) * $this->getItemsPerPage();
  }

  public function exposedFormAlter(&$form, FormStateInterface $form_state) { }

  public function exposedFormValidate(&$form, FormStateInterface $form_state) { }

  public function exposedFormSubmit(&$form, FormStateInterface $form_state, &$exclude) { }

  public function usesExposed() {
    return FALSE;
  }

  protected function itemsPerPageExposed() {
    return FALSE;
  }

  protected function isOffsetExposed() {
    return FALSE;
  }

}

/**
 * @}
 */
